<?php

namespace Citroorder\PaymentGateway;

use Citroorder\PaymentGateway\PaymentGateway as PaymentGateway;
use Citroorder\PaymentGateway\Libraries\Inicis\INIStdPayUtil as INIStdPayUtil;
use Citroorder\PaymentGateway\Libraries\Inicis\HttpClient as HttpClient;
use Exception;

class Inicis extends PaymentGateway
{
    const GATEWAY_TYPE = 'inicis';                                      // PG 종류
    const CANCEL_URL = 'https://iniapi.inicis.com/api/v1/refund';       // 결제취소 URL
    const EXTRA_URL = 'https://iniapi.inicis.com/api/v1/extra';         // 기타API URL

    public $isTestMode = false;                                         // 테스트 모드 여부 (boolean)

    public $version = '1.0';                                            // 전문 버전 ["1.0" 고정]
    public $payMethod = 'card';                                         // 요청지불수단. 빈값으로 세팅 시, 전체 결제수단 표시. 여러지불수단 사용 시 구분자는 ":"(콜론)을 사용. 모바일은 단일선택만 가능.
    public $mid;                                                        // 상점아이디
    public $orderNumber;                                                // 주문번호
    public $price = 0;                                                  // 결제금액
    public $timestamp;                                                  // 타임스템프 [TimeInMillis(Long형)]
    public $signature;                                                  // SHA256 Hash값 [대상: oid, price, timestamp]
    public $mKey;                                                       // SHA256 Hash값 [대상: mid 와 매칭되는 signkey]
    public $currency = 'WON';                                           // 통화구분 ["WON":한화,"USD":달러]
    public $goodName;                                                   // 상품명
    public $buyerName;                                                  // 구매자명
    public $buyerTel;                                                   // 구매자 휴대폰번호
    public $buyerEmail;                                                 // 구매자 이메일주소
    public $returnUrl;                                                  // 결과수신 URL
    public $closeUrl;                                                   // 결제창 닫기 URL
    public $quotaBase;                        // 할부개월수 노출옵션 [할부개월:할부개월]
    public $noInterest;                                                 // 상점부담 무이자할부 옵션 [카드코드-할부개월:할부개월,카드코드-할부개월:할부개월]
    public $tax;                                                        // 부가세
    public $taxFree;                                                    // 비과세 (면세상품 금액)
    public $charset = 'UTF-8';                                          // 결과 수신 인코딩 [UTF-8,EUC-KR]
    public $offerPeriod;                                                // 제공기간 [Y2:년단위결제, M2:월단위결제, YYYYMMDD-YYYYMMDD:시작일-종료일]
    public $payViewType = 'overlay';                                    // 결제창 표시 방법 [overlay,popup]
    public $languageView = 'ko';                                        // 초기 표시 언어 [ko,en,cn]
    public $logoUrl;                                                    // 메인로고 삽입 [size: 89*18]. 이미지 전체 도메인주소 세팅
    public $logoUrl2;                                                   // 서브로고 삽입 [size: 64*13]. 이미지 전체 도메인주소 세팅
    public $popupUrl;                                                   // 팝업처리 URL. payViewType=popup 으로 진행 시 세팅 필요
    public $parentEmail;                                                // 보호자 이메일주소. 14세 미만 필수
    public $iniSsgPayMdn;                                               // SSGPAY 결제요청 시 PUSH전송 휴대폰번호
    public $merchantData;                                               // 가맹점 임의 데이터 (인증결과수신 구간에서 전달). 한글 입력불가, 한글입력은 urlencode 필수
    public $acceptMethod = 'below1000';                                 // 지불수단별 추가옵션. 구분자는 ":"(콜론)을 사용
    public $mobileCharset = 'utf8'; // 모바일 결과 수신 인코딩 [utf8,euc-kr]. Default: euc-kr

    // 지불수단별 추가옵션 설정
    public $acceptMethodOption = array(
        'SKIN' => '#C1272C', // 결제창 배경색상 [Default: #C1272C]
        'below1000' => true, // 신용카드 결제 시 1000원 이하금액 결제가능
        'no_receipt' => true, // 현금영수증 UI 미노출 옵션
    );

    private $signKey; // 가맹점에 제공된 웹 표준 사인키(가맹점 수정후 고정)

    private $SignatureUtil;

    public $sign;

    public $key; // INIAPI key
    public $iv; // INIPAI iv


    public $acceptmethod; // 결제수단별 옵션 (https://manual.inicis.com/stdpay/#acceptmethod)


    public $INIregno; // 주민번호 설정 기능. 13자리(주민번호),10자리(사업자번호),미입력시(화면에서입력가능)


    private $authToken; // 승인요청 검증 토큰 (취소 요청 tid에 따라 유동적)
    private $authUrl; // 승인요청 Url (해당 URL로 HTTPS API Request 승인요청 - POST) 임의 세팅 금지
    private $netCancel; // 망취소요청 Url (승인요청 후 인증결과 수신 실패 / DB저장 실패시) 임의 세팅 금지
    private $tid; // 인증거래번호

    private $testCredentials = [
        'default' => [
            'mid' => 'INIpayTest',
            'signKey' => 'SU5JTElURV9UUklQTEVERVNfS0VZU1RS',
            'key' => 'ItEQKi3rY7uvDS8l',
            'iv' => 'HYb3yQ4f65QL89==',
        ],
        'billing' => [
            'mid' => 'INIBillTst',
            'signKey' => 'SU5JTElURV9UUklQTEVERVNfS0VZU1RS',
            'key' => 'rKnPljRn5m6J9Mzz',
            'iv' => 'W2KLNKra6Wxc1P==',
        ],
        'escrow' => [
            'mid' => 'iniescrow0',
            'signKey' => 'SU5JTElURV9UUklQTEVERVNfS0VZU1RS',
            'key' => 'yERbIlJ3NhTeObsA',
            'iv' => 'tOGDXbfoajk2DQ==',
        ],
    ];

    public function __construct()
    {
        $this->SignatureUtil = new INIStdPayUtil();

        $this->setGateway(static::GATEWAY_TYPE);
        $this->setSiteDomain();

        $this->paymethods = $this->convertPaymethodKeys();

        $this->isMobile();
        if( $this->isMobile === false ) {
            $this->setPaymethod(implode(':', array_keys($this->paymethods['pc'])));
        }
        else {
            $this->setPaymethod(array_keys($this->paymethods['mobile'])[0]);
        }

        // acceptMethod 만들기
        $this->setAcceptMethodOption($this->acceptMethodOption);

        $this->isTestMode(config('paymentgateway.testMode'));
        $this->isStyled(config('paymentgateway.isStyled'));
        $this->hasSummary(config('paymentgateway.hasSummary'));

        $this->setReturnUrl();
        $this->setNotiUrl();

        $this->setTimestamp();

        $this->cardNoInterestQuota = '11-2:3:,34-5:12,14-6:12:24,12-12:36,06-9:12,01-3:4';
        $this->cardQuotaBase = '2:3:4:5:6:11:12:24:36';

        $this->paymentResult = null;
    }

    private function setTimestamp()
    {
        return $this->timestamp = $this->SignatureUtil->getTimestamp();
    }

    public function isTestMode($isTestMode = false, $isTestType = 'default')
    {
        if ( $isTestMode === true ) {
            $this->setCredentials([
                $this->testCredentials[$isTestType]['mid'],
                $this->testCredentials[$isTestType]['signKey'],
                $this->testCredentials[$isTestType]['key'],
                $this->testCredentials[$isTestType]['iv'],
            ]);
        }
        else {
            $this->setCredentials([
                config('paymentgateway.credentials.inicis.mid'),
                config('paymentgateway.credentials.inicis.signKey'),
                config('paymentgateway.credentials.inicis.key'),
                config('paymentgateway.credentials.inicis.iv'),
            ]);
        }

        return $this->isTestMode = $isTestMode;
    }

    public function setCredentials(array $credentials)
    {
        try {
            $this->mid = $credentials ? $credentials[0] : $this->testCredentials['default']['mid'];
            $this->signKey = $credentials ? $credentials[1] : $this->testCredentials['default']['signKey'];
            $this->key = $credentials ? $credentials[2] : $this->testCredentials['default']['key'];
            $this->iv = $credentials ? $credentials[3] : $this->testCredentials['default']['iv'];

            $this->mKey = $this->SignatureUtil->makeHash($this->signKey, 'sha256');
        } catch (Exception $e) {
            echo '문제가 발생했습니다: ' . $e->getMessage();
        }
    }

    public function setPopupUrl($popupUrl = null)
    {
        if( is_null($popupUrl) ) $popupUrl = $this->getValue('siteDomain');
        return $this->popupUrl = $popupUrl;
    }

    public function setCloseUrl($closeUrl = null)
    {
        if( is_null($closeUrl) ) $closeUrl = $this->getValue('siteDomain');
        if( $closeUrl === '_self' ) $closeUrl = $this->getValue('siteDomain') . '/?close';
        return $this->closeUrl = $closeUrl;
    }

    public function setAcceptMethod($acceptmethod)
    {
        return $this->acceptmethod = $acceptmethod;
    }

    public function setOfferPeriod($offerPeriod)
    {
        return $this->offerPeriod = $offerPeriod;
    }

    public function setINIregno($INIregno)
    {
        return $this->INIregno = $INIregno;
    }

    public function setMerchantData($merchantData)
    {
        return $this->merchantData = $merchantData;
    }

    /**
     * 지불수단별 추가옵션 생성
     *
     * @param  string $orderNumber
     * @param  int $price
     * @return string
     */
    public function setAcceptMethodOption ($param = array())
    {
        foreach ($param as $key => $value) {
            if ($key == 'SKIN' && $value) {
                if (!$this->isMobile) {
                    $tempAcceptMethod[] = $key . '(' . $value . ')';
                }
            }

            if ($key == 'below1000' && $value) {
                $tempAcceptMethod[] = $this->isMobile ? $key . '=Y' : $key;
            }

            if ($key == 'no_receipt' && $value) {
                $tempAcceptMethod[] = $this->isMobile ? 'bank_receipt=N' : $key;
            }
        }

        $this->acceptMethod = implode($this->isMobile ? '&' : ':', $tempAcceptMethod);
    }

    /**
     * 위변조 방지체크 signature 생성
     *
     * @param  string $orderNumber
     * @param  int $price
     * @return string
     */
    public function makeSignature (string $orderNumber, int $price)
    {
        return $this->signature = $this->SignatureUtil->makeSignature([
            'oid' => $orderNumber,
            'price' => $price,
            'timestamp' => $this->timestamp,
        ]);
    }

    /**
     * 이니시스 영수증 URL
     * @param String $tid: 승인거래번호
     * @param String $type : recipt(매출전표), cashrecipt(현금영수증)
     */
    public function getReceiptUrl($tid, $type = 'receipt')
    {
        return config('paymentgateway.receiptUrls')[$type] . $tid;
    }

    public function requestPayment()
    {
        $pgData = $this;

        if( $this->isMobile === true ) {
            require_once(__DIR__ . "/../resources/views/{$this->gateway}/mobile/request.php");
        }
        else {

            $this->mKey = $this->SignatureUtil->makeHash($this->signKey, 'sha256');
            $this->sign = $this->SignatureUtil->makeSignature([
                'oid' => $this->orderNumber,
                'price' => $this->price,
                'timestamp' => $this->timestamp,
            ], 'sha256');

            require_once(__DIR__ . "/../resources/views/{$this->gateway}/request.php");
            //return view('paymentgateway::inicis/request')->with('pgData', $this);
        }
    }

    public function returnPayment()
    {
        if( $this->isMobile === true ) {
            return $this->returnPaymentMobile();
        }
        else {
            return $this->returnPaymentPC();
        }
    }

    public function returnPaymentPC()
    {
        $result = [
            'resultCode' => $_REQUEST['resultCode'],
            'resultMsg' => $_REQUEST['resultMsg']
        ];

        $util = $this->SignatureUtil;

        try {

            $httpUtil = new HttpClient();

            if (strcmp('0000', $_REQUEST['resultCode']) == 0) { // 0000이면 성공, 나머지는 실패

                $this->mid = $_REQUEST['mid'];
                $this->signKey = $this->signKey;

                $this->authToken = $_REQUEST['authToken'];
                $this->authUrl = $_REQUEST['authUrl'];
                $this->netCancel = $_REQUEST['netCancelUrl'];

                $this->mKey = hash('sha256', $this->signKey);
                $this->sign = $this->SignatureUtil->makeSignature([
                    'authToken' => $this->authToken,
                    'timestamp' => $this->timestamp,
                ]);

                $authMap = [
                    'mid' => $this->mid,
                    'authToken' => $this->authToken,
                    'signature' => $this->sign,
                    'timestamp' => $this->timestamp,
                    'charset' => $this->charset,
                    'format' => $this->format,
                ];


                    $authResultString = null;
                    if ($httpUtil->processHTTP($this->authUrl, $authMap)) {
                        $authResultString = $httpUtil->body;
                    } else {
                        throw new Exception('결제 통신 에러');
                    }

                    $resultMap = json_decode($authResultString, true);
                    $secureSignature = $this->SignatureUtil->makeSignatureAuth([
                        'mid' => $this->mid,
                        'tstamp' => $this->timestamp,
                        'MOID' => $resultMap['MOID'],
                        'TotPrice' => $resultMap['TotPrice'],
                    ]);

                    //*** 여기까지 왔으면 결제 보안 통과. 아래 거래 결과 처리 ***//

                    if ((strcmp('0000', $resultMap['resultCode']) == 0) && (strcmp($secureSignature, $resultMap['authSignature']) == 0) ){
                        /**
                         *
                         * [ 거래 성공 ]
                         *
                         */
                        $result = $resultMap;
                        $result['resultCode'] = $resultMap['resultCode'];
                        $result['resultMsg'] = $resultMap['resultMsg'];
                        $this->paymentResult = $resultMap;
                    }
                    else {
                        /**
                         *
                         * [ 거래 실패 ]
                         *
                         */
                        if (strcmp($secureSignature, $resultMap['authSignature']) != 0) {
                            if(strcmp('0000', $resultMap['resultCode']) == 0) {
                                throw new Exception('데이터 위변조 체크 실패');
                            }
                        }

                        throw new Exception('거래실패 : ' . $resultMap['resultMsg']);


                    }


            }
            else {
                // 인증실패
                throw new Exception('인증실패 : ' . $_REQUEST['resultMsg']);
            }
        } catch (Exception $e) {

            $this->netCancel();

            $result = [
                'resultCode' => $e->getCode(),
                'resultMsg' => $e->getMessage(),
            ];
        } finally {
            return $result;
        }

        // $pgData = $result['resultCode'] == '0000' ? $this->paymentResult : json_decode(json_encode($result), FALSE);
        // require_once(__DIR__ . "/../resources/views/{$this->gateway}/return.php");
    }

    public function returnPaymentMobile()
    {
        $result = [
            'resultCode' => $_REQUEST['P_STATUS'],
            'resultMsg' => $_REQUEST['P_RMESG1']
        ];

        try {
            if (isset($_REQUEST['P_STATUS']) && ($_REQUEST['P_STATUS'] === '00')) {

                $data = array(
                    'P_MID' => substr($_REQUEST['P_TID'], '10', '10'), // P_MID
                    'P_TID' => $_REQUEST['P_TID'], // P_TID
                );

                // curl 통신 시작
                $ch = curl_init(); // curl 초기화
                curl_setopt($ch, CURLOPT_URL, $_REQUEST['P_REQ_URL']); // URL 지정하기
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); // 요청 결과를 문자열로 반환
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // connection timeout 10초
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0); // 원격 서버의 인증서가 유효한지 검사 안함
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data)); // POST 로 $data 를 보냄
                curl_setopt($ch, CURLOPT_POST, 1); // true시 post 전송

                $response = curl_exec($ch);

                if (curl_errno($ch)){
                    throw new Exception('결제 통신 오류'); // curl_error($ch)
                }

                curl_close($ch);

                $returnArr = explode('&' , $response);

                $resultMap = [];
                foreach ($returnArr as $value) {
                    $tempArr = explode('=' , $value);
                    $resultMap[$tempArr[0]] = $tempArr[1];
                }

                // 승인 성공
                if ($resultMap['P_STATUS'] === '00') {
                    $result = $resultMap;
                    $result['resultCode'] = $resultMap['P_STATUS'];
                    $result['resultMsg'] = $resultMap['P_RMESG1'];
                    $this->paymentResult = $resultMap;
                } else {
                    throw new Exception('거래실패 : ' . $resultMap['P_RMESG1']);
                }
            }
            else {
                // 인증실패
                throw new Exception('인증실패 : ' . $_REQUEST['P_RMESG1']);
            }
        } catch (Exception $e) {

            $this->netCancelMobile();

            $result = [
                'resultCode' => $e->getCode(),
                'resultMsg' => $e->getMessage(),
            ];
        } finally {
            return $result;
        }
    }

    // 망취소(PC)
    public function netCancel()
    {
        $result = '';
        $util = $this->SignatureUtil;

        $netCancel = $_REQUEST['netCancelUrl'];

        if ($netCancel) {
            if ($_REQUEST['mid'] && $_REQUEST['authToken']) {
                $mid = $_REQUEST['mid'];
                $authToken = $_REQUEST['authToken'];
                $timestamp = $util->getTimestamp();

                // signature 데이터 생성 (모듈에서 자동으로 signParam을 알파벳 순으로 정렬후 NVP 방식으로 나열해 hash)
                $signature = $util->makeSignature([
                    'authToken' => $authToken, // 필수
                    'timestamp' => $timestamp, // 필수
                ]);

                $cancelMap = [
                    'mid' => $mid,
                    'authToken' => $authToken,
                    'signature' => $signature,
                    'timestamp' => $timestamp,
                    'charset' => $this->charset,
                    'format' => $this->format,
                ];

                // 거래 내역이 있는지 상관없이 망취소 요청
                // curl 통신 시작
                $ch = curl_init(); // curl 초기화
                curl_setopt($ch, CURLOPT_URL, $netCancel); // URL 지정하기
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); // 요청 결과를 문자열로 반환
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // connection timeout 10초
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0); // 원격 서버의 인증서가 유효한지 검사 안함
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($cancelMap)); // POST 로 $data 를 보냄
                curl_setopt($ch, CURLOPT_POST, 1);                                         //true시 post 전송

                $result = curl_exec($ch);

                curl_close($ch);
            }
        }

        return json_decode($result, true);
    }

    // 망취소(모바일)
    public function netCancelMobile()
    {
        $result = [];
        $authUrl = $_REQUEST['P_REQ_URL'];

        if ($authUrl) {

            $parseAuthUrl = parse_url($authUrl);

            // 망취소 요청 url
            $netCancel = $parseAuthUrl['scheme'] . '://' . $parseAuthUrl['host'] . '/smart/payNetCancel.ini';

            if ($_REQUEST['P_TID'] && $_REQUEST['P_AMT'] && $_REQUEST['P_NOTI']) {
                $cancelMap = [
                    'P_TID' => $_REQUEST['P_TID'],
                    'P_MID' => substr($_REQUEST['P_TID'], '10', '10'),
                    'P_AMT' => $_REQUEST['P_AMT'],
                    'P_OID' => $_REQUEST['P_NOTI'],
                ];

                // 거래 내역이 있는지 상관없이 망취소 요청
                // curl 통신 시작
                $ch = curl_init(); // curl 초기화
                curl_setopt($ch, CURLOPT_URL, $netCancel); // URL 지정하기
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); // 요청 결과를 문자열로 반환
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10); // connection timeout 10초
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0); // 원격 서버의 인증서가 유효한지 검사 안함
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($cancelMap)); // POST 로 $data 를 보냄
                curl_setopt($ch, CURLOPT_POST, 1);                                         //true시 post 전송

                $response = curl_exec($ch);

                $returnArr = explode('&' , iconv('EUC-KR', 'UTF-8', $response));

                foreach ($returnArr as $value) {
                    $tempArr = explode('=' , $value);
                    $result[$tempArr[0]] = $tempArr[1];
                }

                curl_close($ch);
            }
        }

        return $result;

    }

    public function returnVBankPayment()
    {

    }

    /**
     * 결제 취소
     * https://manual.inicis.com/iniapi/cancel.html#cancel
     *
     * @param array $param
     * paymethod: 지불수단 (필수)
     * tid: 거래번호 (필수)
     * msg: 취소사유 (옵션)
     * @return array
     * resultCode: 결과코드["00":성공, 그외 실패]
     * resultMsg: 결과메세지
     * cancelDate: 취소일자 [YYYYMMDD]
     * cancelTime: 취소시간 [hhmmss]
     * cshrCancelNum: 현금영수증 취소승인번호 (현금영수증 발행건에 한함)
     */
    public function cancel(
        $param = [
            'paymethod' => null,
            'tid' => null,
            'msg' => '결제 취소',
        ]
    )
    {
        try {
            // 필수항목 검사 (지불수단코드 - paymethod)
            if (!isset($param['paymethod']) || empty($param['paymethod'])) {
                throw new Exception('지불수단 코드는 필수항목 입니다.');
            }

            // 필수항목 검사 (거래번호 - tid)
            if (!isset($param['tid']) || empty($param['tid'])) {
                throw new Exception('TID는 필수항목 입니다.');
            }

            // 취소 데이터 준비
            $type = 'Refund';
            $paymethod = $param['paymethod'];
            $timestamp = date('YmdHis');
            $clientIp = $_SERVER['REMOTE_ADDR'];
            $mid = $this->mid;
            $tid = $param['tid'];
            $msg = $param['msg'];
            $hashData = hash('sha512', $this->key . $type . $paymethod . $timestamp . $clientIp . $mid . $tid);

            $authMap = [
                'type' => $type,
                'paymethod' => $paymethod,
                'timestamp' => $timestamp,
                'clientIp' => $clientIp,
                'mid' => $mid,
                'tid' => $tid,
                'msg' => $msg,
                'hashData' => $hashData,
            ];

            $httpUtil = new HttpClient();

            if ($httpUtil->processHTTP(static::CANCEL_URL, $authMap)) {
                $authResultString = $httpUtil->body;

                $resultMap = json_decode($authResultString, true);

                // 인증 성공
                if (strcmp('00', $resultMap['resultCode']) == 0) {
                    $result = $resultMap;
                } else {
                    throw new Exception($resultMap['resultMsg']);
                }
            } else {
                throw new Exception('결제 취소 통신 에러');
            }
        } catch (Exception $e) {
            $result = [
                'resultCode' => $e->getCode(),
                'resultMsg' => $e->getMessage(),
            ];
        } finally {
            return $result;
        }
    }

    /**
     * 부분 결제 취소
     * https://manual.inicis.com/iniapi/cancel.html#cancel
     *
     * @param array $param
     * paymethod: 지불수단 (필수)
     * tid: 거래번호 (필수)
     * msg: 취소사유 (옵션)
     * price: 취소요청금액 (필수)
     * confirmPrice: 부분취소 후 남은금액 (필수)
     * @return array
     * resultCode: 결과코드["00":성공, 그외 실패]
     * resultMsg: 결과메세지
     * tid: 부분취소TID
     * prtcTid: 원거래TID
     * prtcPrice: 부분취소금액
     * prtcRemains: 부분취소 후 남은금액
     * prtcCnt: 부분취소 요청 횟수
     * prtcType: 부분취소 구분 ["0":재승인방식, "1":부분취소]
     * prtcDate: 부분취소일자 [YYYYMMDD]
     * prtcTime: 부분취소시간 [hhmmss]
     * receiptInfo:
     */
    public function cancelPart(
        $param = [
            'paymethod' => null,
            'tid' => null,
            'msg' => '부분 결제 취소',
            'price' => null,
            'confirmPrice' => null,
        ]
    )
    {
        try {
            // 필수항목 검사 (지불수단코드 - paymethod)
            if (!isset($param['paymethod']) || empty($param['paymethod'])) {
                throw new Exception('지불수단 코드는 필수항목입니다.');
            }

            // 필수항목 검사 (거래번호 - tid)
            if (!isset($param['tid']) || empty($param['tid'])) {
                throw new Exception('거래번호 TID는 필수항목입니다.');
            }

            // 필수항목 검사 (취소요청금액 - price)
            if (!isset($param['price'])) {
                throw new Exception('취소요청금액은 필수항목입니다.');
            }

            if ($param['price'] <= 0) {
                throw new Exception('취소요청금액이 0원 이하입니다.');
            }

            // 필수항목 검사 (부분취소 후 남은금액 - confirmPrice)
            if (!isset($param['confirmPrice'])) {
                throw new Exception('부분취소 후 남은금액은 필수항목입니다.');
            }

            if ($param['confirmPrice'] < 0) {
                throw new Exception('부분취소 후 남은금액 오류입니다.');
            }

            // 부분 취소 데이터 준비
            $type = 'PartialRefund';
            $paymethod = $param['paymethod'];
            $timestamp = date('YmdHis');
            $clientIp = $_SERVER['REMOTE_ADDR'];
            $mid = $this->mid;
            $tid = $param['tid'];
            $msg = $param['msg'];
            $price = $param['price'];
            $confirmPrice = $param['confirmPrice'];

            $hashData = hash('sha512', $this->key . $type . $paymethod . $timestamp . $clientIp . $mid . $tid . $price . $confirmPrice);

            $authMap = [
                'type' => $type,
                'paymethod' => $paymethod,
                'timestamp' => $timestamp,
                'clientIp' => $clientIp,
                'mid' => $mid,
                'tid' => $tid,
                'msg' => $msg,
                'price' => $price,
                'confirmPrice' => $confirmPrice,
                'hashData' => $hashData,
            ];

            $httpUtil = new HttpClient();

            if ($httpUtil->processHTTP(static::CANCEL_URL, $authMap)) {
                $authResultString = $httpUtil->body;

                $resultMap = json_decode($authResultString, true);

                // 인증 성공
                if (strcmp('00', $resultMap['resultCode']) == 0) {
                    $result = $resultMap;
                } else {
                    throw new Exception($resultMap['resultMsg']);
                }
            } else {
                throw new Exception('부분 결제 취소 통신 에러');
            }

        } catch (Exception $e) {
            $result = [
                'resultCode' => $e->getCode(),
                'resultMsg' => $e->getMessage(),
            ];
        } finally {
            return $result;
        }
    }

     /**
     * 거래 조회
     * https://manual.inicis.com/iniapi/extra.html#Inquiry
     *
     * @param array $param
     * originalTid: 거래번호 (필수)
     * @return array
     * resultCode: 결과코드["00":성공, 그외 실패]
     * resultMsg: 결과메세지
     * tid: 거래TID
     * price: 거래금액
     * status: 거래상태 ["0":승인, "1":취소, "9":거래없음] 가상계좌 거래 시 ["N":입금대기, "Y":입금완료, "C":입금 전 취소]
     * paymethod: 지불수단 * 해외결제만 전달
     * oid: 상점 주문번호 * 해외결제만 전달
     * applNum: 승인번호 * 해외결제만 전달
     * priceExchange: 원화승인금액 * 해외결제만 전달
     * rtExchange: 환율 * 해외결제만 전달 (은련카드 제외)
     * applDate: 승인일자 * ISP결제만 전달
     * applTime: 승인시간 * ISP결제만 전달
     *
     */
    public function inquiry(
        $param = [
            'originalTid' => null,
        ]
    )
    {
        try {
            // 필수항목 검사 (거래번호 - tid)
            if (!isset($param['originalTid']) || empty($param['originalTid'])) {
                throw new Exception('거래번호 TID는 필수항목입니다.');
            }

            $type = 'Extra';
            $paymethod = 'Inquiry';
            $timestamp = date('YmdHis');
            $clientIp = $_SERVER['REMOTE_ADDR'];
            $mid = $this->mid;
            $originalTid = $param['originalTid'];
            $hashData = hash('sha512', $this->key . $type . $paymethod . $timestamp . $clientIp . $mid);

            $authMap = [
                'type' => $type,
                'paymethod' => $paymethod,
                'timestamp' => $timestamp,
                'clientIp' => $clientIp,
                'mid' => $mid,
                'originalTid' => $originalTid,
                'hashData' => $hashData,
            ];

            $httpUtil = new HttpClient();

            if ($httpUtil->processHTTP(static::EXTRA_URL, $authMap)) {
                $authResultString = $httpUtil->body;

                $resultMap = json_decode($authResultString, true);

                // 인증 성공
                if (strcmp('00', $resultMap['resultCode']) == 0) {
                    $result = $resultMap;
                } else {
                    throw new Exception($resultMap['resultMsg']);
                }
            } else {
                throw new Exception('거래조회 통신 에러');
            }
        } catch (Exception $e) {
            $result = [
                'resultCode' => $e->getCode(),
                'resultMsg' => $e->getMessage(),
            ];
        } finally {
            return $result;
        }
    }
}
