<?php

namespace Citroorder\PaymentGateway;

use Citroorder\PaymentGateway\Libraries\MobileDetect;
use Citroorder\PaymentGateway\Libraries\Encoding;

class PaymentGateway
{

    public $gateway; // PG 종류
    public $isMobile = false; // 모바일 결제 여부 (boolean)
    public $setMobile = false; // 모바일 결제 강제 세팅 (boolean)
    public $isStyled = true; // 기본 CSS 적용 여부 (boolean)
    public $hasSummary = true; // 요약표 보이기 여부 (boolean)

    public $language = 'ko'; // 결제창 표시 언어 [ko: 한국어, en: 영어, cn: 중국어]
    public $charset = 'UTF-8'; // 결과 수신 인코딩 [UTF-8, EUC-KR]
    public $format = 'JSON'; // 리턴형식 (XML, JSON, NVP)
    public $siteDomain; // 사이트 도메인
    public $returnUrl; // 결제 완료 후 리턴 URL
    public $notiUrl; // 가상계좌입금통보 URL

    public $price = 0; // 결제할 금액
    public $goodname = ''; // 상품명 (40Byte 초과 요청시 37Byte + ...으로 자동 변환)
    public $currency = 'WON'; // 통화구분 (WON:한화,USD:달러 [USD는 카드 결제만 가능])
    public $tax; // 부가세 ("부가세 업체정함" 계약가맹점만 설정필요) 주의: 전체금액의 10%이하로 설정
    public $taxfree; // 비과세 ("부가세 업체정함" 계약가맹점만 설정필요)
    public $paymethod; // 요청결제수단 (Card, VBank ...)
    public $paymethods = []; // 결제수단 리스트 (config/paymentgateway에서 설정)
    public $nointerest; // 무이자할부
    public $quotabase; // 할부

    public $orderNumber; // 주문번호
    public $buyername = ''; // 구매자명 (30Byte 초과 요청시 30Byte로 자동 변환)
    public $buyertel = ''; // 구매자 휴대폰번호 (숫자와 "-" 만 허용)
    public $buyeremail = ''; // 구매자 Email ("@", "." 외 특수문자 입력불가)
    public $parentemail = ''; // 보호자 Email (14세 미만 필수, "@", "." 외 특수문자 입력불가)

    public $customFields = [];

    public $paymentResult; // 결제 승인 응답 데이터

    public function runPaymentGateway(array $array)
    {
        foreach( $array as $function => $data ) {
            $this->$function($data);
        }
    }

    public function setGateway($gateway)
    {
        return $this->gateway = $gateway;
    }

    public function setMobile($setMobile = false)
    {
        $this->setMobile = $setMobile;
        $this->isMobile();
    }

    public function isMobile()
    {
        $mobileDetect = new MobileDetect();
        $this->isMobile = $this->setMobile;
        if( $this->setMobile === false ) $this->isMobile = $mobileDetect->isMobile();

        return $this->isMobile;
    }

    public function isStyled($isStyled)
    {
        return $this->isStyled = $isStyled;
    }

    public function printReturnMessage($string = '')
    {
        return (mb_check_encoding($string, 'UTF-8') === true) ? $string : iconv('EUC-KR', 'UTF-8', $string);
        //return Encoding::fixUTF8($string);
    }

    public function hasSummary($hasSummary)
    {
        return $this->hasSummary = $hasSummary;
    }

    public function setSiteDomain($siteDomain = null)
    {
        if ( is_null($siteDomain) ) {
            if( isset( $_SERVER['HTTP_HOST'] ) ) { // PHP CLI로 접근시 발생하는 Undefined Index 에러를 위한 예외처리
                $url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . "://{$_SERVER['HTTP_HOST']}{$_SERVER['REQUEST_URI']}";
                $urlParts = parse_url($url);
                $siteDomain = $urlParts['scheme'] . '://' . preg_replace('/^www\./', '', ($urlParts['host'] ?? ''));// . $urlParts['path'];
            }

            if (! preg_match('#^http(s)?://#', $siteDomain)) { $siteDomain = 'http://' . $siteDomain; }
            $siteDomain = rtrim($siteDomain, '/\\');
        }
        else { // $_SERVER 변수에서 넘겨줄 값이 없으면, PHP CLI 접근으로 판단하고 localhost 반환
            $siteDomain = 'http://localhost';
        }

        return $this->siteDomain = $siteDomain;
    }

    public function setLanguage($language)
    {
        return $this->language = $language;
    }

    public function setReturnUrl($returnUrl = null)
    {
        if ( is_null($returnUrl) ) $returnUrl = $this->getValue('siteDomain') . '/pg/return';
        return $this->returnUrl = $returnUrl;
    }

    public function setNotiUrl($notiUrl = null)
    {
        if ( is_null($notiUrl) ) $notiUrl = $this->getValue('siteDomain') . '/pg/noti';
        return $this->notiUrl = $notiUrl;
    }

    public function setOrderNumber($orderNumber)
    {
        return $this->orderNumber = $orderNumber;
    }

    public function setPrice($price, $priceType = 'price')
    {
        $price = preg_replace('/\D/', '', $price);
        if( in_array($priceType, ['price', 'tax', 'taxFree']) ) return $this->{$priceType} = $price;
    }

    public function setGoodName($goodname)
    {
        return $this->goodname = $goodname;
    }

    public function setBuyer(array $buyer)
    {
        $this->buyername = $buyer['name'];
        $this->buyertel = $buyer['tel'];
        $this->buyeremail = isset($buyer['email']) ? $buyer['email'] : null;
        $this->parentemail = isset($buyer['parentemail']) ? $buyer['parentemail'] : null;
        return true;
    }

    /**
     * 요청결제수단(빈값으로 세팅 시, 전체 결제수단 표시
     */
    public function setPaymethod($paymethod)
    {
        return $this->paymethod = $paymethod;
    }

    public function setCardNoInterestQuota($nointerest)
    {
        return $this->nointerest = $nointerest;
    }

    public function setCardQuotaBase($quotabase)
    {
        return $this->quotabase = $quotabase;
    }

    public function setCharset($charset = 'UTF-8')
    {
        return $this->charset = $charset;
    }

    public function setFormat($format = 'JSON')
    {
        return $this->format = $format;
    }

    public function convertPaymethodKeys()
    {
        $paymethodKeys = [
            'inicis' => [
                'pc' => [
                    'card' => 'Card',
                    'bank' => 'DirectBank',
                    'vbank' => 'VBank',
                ],
                'mobile' => [
                    'card' => 'CARD',
                    'bank' => 'BANK',
                    'vbank' => 'VBANK',
                ],
            ],
        ];

        $result = [];
        foreach( config('paymentgateway.paymethods') as $platform => $items ) {
            foreach( $items as $key => $value ) {
                $result[$platform][$paymethodKeys[$this->gateway][$platform][$key]] = $value;
            }
        }

        return $result;
    }

    public function setCustomFields($customFields)
    {
        return $this->customFields = $customFields;
    }

    public function getCustomFields()
    {
        return config('paymentgateway.customFields') ?: $this->customFields;
    }

    public function getValue($variableName)
    {
        return $this->{$variableName};
    }

}
